#include "rc_hash.h"

#include "../rhash/rc_hash_internal.h"

#include "../rc_compat.h"
#include "../test_framework.h"
#include "data.h"
#include "mock_filereader.h"

#include <stdlib.h>

static const uint8_t data_dosz[] = {
  'P','K',0x03,0x04, /* local file header signature */
  0x14,0x00, 0x02,0x00, 0x08,0x00, 0x00,0xBC, 0x98,0x21, 0x00,0x00,0x00,0x00, 0x02,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, /* info */
  0x07,0x00, 0x00,0x00, 'F','O','L','D','E','R','/', /* file name length, extra length, name data */
  0x03, 0x00, /* compressed content */

  'P','K',0x03,0x04, /* local file header signature */
  0x14,0x00, 0x02,0x00, 0x08,0x00, 0x00,0xBC, 0x98,0x21, 0x31,0xCF,0xD0,0x4A, 0x03,0x00,0x00,0x00, 0x01,0x00,0x00,0x00, /* info */
  0x0E,0x00, 0x00,0x00, 'F','O','L','D','E','R','/','S','U','B','.','T','X','T', /* file name length, extra length, name data */
  0x73, 0x02, 0x00, /* compressed content */

  'P','K',0x03,0x04, /* local file header signature */
  0x14,0x00, 0x02,0x00, 0x08,0x00, 0x00,0xBC, 0x98,0x21, 0x8B,0x9E,0xD9,0xD3, 0x03,0x00,0x00,0x00, 0x01,0x00,0x00,0x00, /* info */
  0x08,0x00, 0x00,0x00, 'R','O','O','T','.','T','X','T', /* file name length, extra length, name data */
  0x73, 0x04, 0x00, /* compressed content */

  'P', 'K',0x01,0x02, /* central directory file header signature */
  0x00,0x00, 0x14,0x00, 0x02,0x00, 0x08,0x00, 0x00,0xBC, 0x98,0x21, 0x00,0x00,0x00,0x00, 0x02,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
  0x07,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
  'F','O','L','D','E','R','/',

  'P','K',0x01,0x02, /* central directory file header signature */
  0x00,0x00, 0x14,0x00, 0x02,0x00, 0x08,0x00, 0x00,0xBC, 0x98,0x21, 0x31,0xCF,0xD0,0x4A, 0x03,0x00,0x00,0x00, 0x01,0x00,0x00,0x00,
  0x0E,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00,0x00,0x00, 0x27,0x00,0x00,0x00,
  'F','O','L','D','E','R','/','S','U','B','.','T','X','T',

  'P','K',0x01,0x02, /* central directory file header signature */
  0x00,0x00, 0x14,0x00, 0x02,0x00, 0x08,0x00, 0x00,0xBC, 0x98,0x21, 0x8B,0x9E,0xD9,0xD3, 0x03,0x00,0x00,0x00, 0x01,0x00,0x00,0x00,
  0x08,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00,0x00,0x00, 0x56,0x00,0x00,0x00,
  'R','O','O','T','.','T','X','T',

  'P','K',0x05,0x06, /* end of central directory signature */
  0x00,0x00, 0x00,0x00, /* disk number */
  0x03,0x00, 0x03,0x00, /* number of directory entries on this disk and total */
  0xA7,0x00,0x00,0x00, /* size of central directory (bytes) */
  0x7F,0x00,0x00,0x00, /* offset of start of central directory */
  0x16,0x00, 'T','O','R','R','E','N','T','Z','I','P','P','E','D','-','F','D','0','7','C','5','2','C' /* comment length and comment data */
};

static const uint8_t data_dosz_zip64[] = {
  'P','K',0x03,0x04, /* local file header signature */
  0x14,0x00, 0x00,0x00, 0x08,0x00, 0x30,0x74, 0x0A,0x41, 0x7E,0xE7,0xFF,0x69, 0x24,0x00,0x00,0x00, 0x24,0x00,0x00,0x00,
  0x06,0x00, 0x00,0x00, 'R','E','A','D','M','E',
  0x0B,0xC9,0xC8,0x2C,0x56,0x28,0xCE,0x4D,0xCC,0xC9,0x51,0x48,0xCB,0xCC,0x49,0x55,0x00,0xF2,
  0x32,0xF3,0x14,0xA2,0x3C,0x03,0xCC,0x4C,0x14,0xD2,0xF2,0x8B,0x72,0x13,0x4B,0xF4,0xB8,0x00,

  'P','K',0x01,0x02, /* central directory file header signature */
  0x2D,0x03, 0x2D,0x00, 0x00,0x00, 0x08,0x00, 0x30,0x74, 0x0A,0x41, 0x7E,0xE7,0xFF,0x69, 0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0xFF,0xFF,
  0x06,0x00, 0x14,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00,0xA4,0x81, 0x00,0x00,0x00,0x00,
  'R','E','A','D','M','E',
  0x01,0x00, /* Zip64 extended information extra field header id */
  0x10,0x00, /* Size of extra field chunk */
  0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* uncompressed file size */
  0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* size of compressed data */

  'P','K',0x06,0x06, /* Zip64 End of central directory record signature */
  0x2C,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* Size of the EOCD64 minus 12 */
  0x2D,0x00, 0x2D,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x48,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x48,0x00,0x00,0x00,0x00,0x00,0x00,0x00,

  'P','K',0x06,0x07, /* end of central dir locator signature */
  0x00,0x00,0x00,0x00, /* number of the disk with the start of the zip64 end of central directory */
  0x90,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* relative offset of the zip64 end of central directory record */
  0x01,0x00,0x00,0x00, /* total number of disks */

  'P','K',0x05,0x06, /* end of central directory signature */
  0x00,0x00, 0x00,0x00, /* disk number */
  0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF, /* file and size numbers are 0xFF in Zip64 */
  0x00,0x00 /* comment length */
};

static const uint8_t data_child_dosz[] = {
  'P','K',0x03,0x04, /* local file header signature */
  0x0A,0x00, 0x00,0x00, 0x00,0x00, 0x00,0xBC, 0x98,0x21, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, /* info */
  0x10,0x00, 0x00,0x00, 'b','a','s','e','.','d','o','s','z','.','p','a','r','e','n','t', /* file name length; extra length; name data */
  0x73, 0x04, 0x00, /* compressed content */

  'P','K',0x03,0x04,
  0x0A,0x00, 0x00,0x00, 0x00,0x00, 0x00,0xBC, 0x98,0x21, 0x29,0x54,0xB3,0x22, 0x05,0x00,0x00,0x00, 0x05,0x00,0x00,0x00, /* info */
  0x09,0x00, 0x00,0x00, 'C','H','I','L','D','.','T','X','T', /* file name length; extra length; name data */
  0x63, 0x68, 0x69, 0x6C, 0x64, /* content */

  'P','K',0x01,0x02, /* central directory file header signature */
  0x3F,0x00, 0x0A,0x00, 0x00,0x00, 0x00,0x00, 0x00,0xBC, 0x98,0x21, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
  0x10,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x20,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
  'b','a','s','e','.','d','o','s','z','.','p','a','r','e','n','t',

  'P','K',0x01,0x02, /* central directory file header signature */
  0x3F,0x00, 0x0A,0x00, 0x00,0x00, 0x00,0x00, 0x00,0xBC, 0x98,0x21, 0x29,0x54,0xB3,0x22, 0x05,0x00,0x00,0x00, 0x05,0x00,0x00,0x00,
  0x09,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00, 0x20,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
  'C','H','I','L','D','.','T','X','T',

  'P','K',0x05,0x06, /* end of central directory signature */
  0x00,0x00, 0x00,0x00, /* disk number */
  0x02,0x00, 0x02,0x00, /* number of directory entries on this disk and total */
  0x75,0x00,0x00,0x00, /* size of central directory (bytes) */
  0x5D,0x00,0x00,0x00, /* offset of start of central directory */
  0x00,0x00 /* comment length */
};

static void test_hash_msdos_dosz()
{
  char hash_file[33], hash_iterator[33];
  const char* expected_md5 = "59a255662262f5ada32791b8c36e8ea7";

  mock_file(0, "game.dosz", data_dosz, sizeof(data_dosz));

  /* test file hash */
  int result_file = rc_hash_generate_from_file(hash_file, RC_CONSOLE_MS_DOS, "game.dosz");

  /* test file identification from iterator */
  struct rc_hash_iterator iterator;
  rc_hash_initialize_iterator(&iterator, "game.dosz", NULL, 0);
  int result_iterator = rc_hash_iterate(hash_iterator, &iterator);
  rc_hash_destroy_iterator(&iterator);

  /* validation */
  ASSERT_NUM_EQUALS(result_file, 1);
  ASSERT_STR_EQUALS(hash_file, expected_md5);

  ASSERT_NUM_EQUALS(result_iterator, 1);
  ASSERT_STR_EQUALS(hash_iterator, expected_md5);
}

static void test_hash_msdos_dosz_zip64()
{
  char hash_file[33];
  const char* expected_md5 = "927dad0a57a2860267ab7bcdb8bc3f61";

  mock_file(0, "game.dosz", data_dosz_zip64, sizeof(data_dosz_zip64));

  /* test file hash */
  int result_file = rc_hash_generate_from_file(hash_file, RC_CONSOLE_MS_DOS, "game.dosz");

  /* validation */
  ASSERT_NUM_EQUALS(result_file, 1);
  ASSERT_STR_EQUALS(hash_file, expected_md5);
}

static void test_hash_msdos_dosz_with_dosc()
{
  char hash_dosc[33];
  const char* expected_dosc_md5 = "dd0c0b0c170c30722784e5e962764c35";

  /* Add main dosz file and overlay dosc file which will get hashed together */
  mock_file(0, "game.dosz", data_dosz, sizeof(data_dosz));
  mock_file(1, "game.dosc", data_dosz, sizeof(data_dosz));

  /* test file hash */
  int result_dosc = rc_hash_generate_from_file(hash_dosc, RC_CONSOLE_MS_DOS, "game.dosz");

  /* validation */
  ASSERT_NUM_EQUALS(result_dosc, 1);
  ASSERT_STR_EQUALS(hash_dosc, expected_dosc_md5);
}

static void test_hash_msdos_dosz_with_parent()
{
  char hash_dosz[33], hash_dosc[33], hash_dosc2[33];
  const char* expected_dosz_md5 = "623c759476b8b5adb46362f8f0b60769";
  const char* expected_dosc_md5 = "ecd9d776cbaad63094829d7b8dbe5959";
  const char* expected_dosc2_md5 = "cb55c123936ad84479032ea6444cb1a1";

  /* Add base dosz file and child dosz file which will get hashed together */
  mock_file(0, "base.dosz", data_dosz, sizeof(data_dosz));
  mock_file(1, "child.dosz", data_child_dosz, sizeof(data_child_dosz));

  /* test file hash */
  int result_dosz = rc_hash_generate_from_file(hash_dosz, RC_CONSOLE_MS_DOS, "child.dosz");

  /* test file hash with base.dosc also existing */
  mock_file(2, "base.dosc", data_dosz, sizeof(data_dosz));
  int result_dosc = rc_hash_generate_from_file(hash_dosc, RC_CONSOLE_MS_DOS, "child.dosz");

  /* test file hash with child.dosc also existing */
  mock_file(3, "child.dosc", data_dosz, sizeof(data_dosz));
  int result_dosc2 = rc_hash_generate_from_file(hash_dosc2, RC_CONSOLE_MS_DOS, "child.dosz");

  /* validation */
  ASSERT_NUM_EQUALS(result_dosz, 1);
  ASSERT_NUM_EQUALS(result_dosc, 1);
  ASSERT_NUM_EQUALS(result_dosc2, 1);
  ASSERT_STR_EQUALS(hash_dosz, expected_dosz_md5);
  ASSERT_STR_EQUALS(hash_dosc, expected_dosc_md5);
  ASSERT_STR_EQUALS(hash_dosc2, expected_dosc2_md5);
}

/* ========================================================================= */

void test_hash_zip(void) {
  TEST_SUITE_BEGIN();

  /* MS DOS */
  TEST(test_hash_msdos_dosz);
  TEST(test_hash_msdos_dosz_zip64);
  TEST(test_hash_msdos_dosz_with_dosc);
  TEST(test_hash_msdos_dosz_with_parent);

  TEST_SUITE_END();
}
