// Copyright (c) 2022 CTTC
// Copyright (c) 2023 Centre Tecnologic de Telecomunicacions de Catalunya (CTTC)
//
// SPDX-License-Identifier: GPL-2.0-only

#ifndef TRAFFIC_GENERATOR_TEST_H
#define TRAFFIC_GENERATOR_TEST_H

#include "ns3/abort.h"
#include "ns3/config.h"
#include "ns3/inet-socket-address.h"
#include "ns3/internet-stack-helper.h"
#include "ns3/ipv4-address-helper.h"
#include "ns3/ipv4-global-routing-helper.h"
#include "ns3/log.h"
#include "ns3/packet-sink-helper.h"
#include "ns3/packet-sink.h"
#include "ns3/simple-channel.h"
#include "ns3/simple-net-device.h"
#include "ns3/simulator.h"
#include "ns3/string.h"
#include "ns3/test.h"
#include "ns3/three-gpp-http-variables.h"
#include "ns3/traffic-generator-helper.h"
#include "ns3/traffic-generator-ngmn-ftp-multi.h"
#include "ns3/traffic-generator-ngmn-gaming.h"
#include "ns3/traffic-generator-ngmn-video.h"
#include "ns3/traffic-generator-ngmn-voip.h"
#include "ns3/uinteger.h"

#include <fstream>
#include <list>

namespace ns3
{

/**
 * @ingroup applications
 * @defgroup applications-test applications module tests
 */

/**
 * @ingroup applications-test
 * @ingroup tests
 *
 * Test that all the packets generated by the client application (e.g., NGMN VoIP,
 * NGMN VIDEO, NGMN GAMING, NGMN FTP, etc.) are correctly received by a server
 * application and tests for the different protocols (TCP/UCP).
 */
class TrafficGeneratorTestCase : public TestCase
{
  public:
    TrafficGeneratorTestCase(std::string name,
                             TypeId trafficGeneratorType,
                             std::string transportProtocol);
    ~TrafficGeneratorTestCase() override;

  private:
    void DoRun() override;
    TypeId m_trafficGeneratorType;   //!< the traffic generator application to be tested: NGMN VoIP,
                                     //!< NGMN VIDEO, NGMN GAMING, NGMN FTP
    std::string m_transportProtocol; //!< the transport protocol to be used: TCP or UDP
};

/**
 * @ingroup applications-test
 * @ingroup tests
 *
 * Test that the file size is generated according to NGMN parameters
 */
class TrafficGeneratorNgmnFtpTestCase : public TestCase
{
  public:
    TrafficGeneratorNgmnFtpTestCase();
    ~TrafficGeneratorNgmnFtpTestCase() override;

  private:
    void DoRun() override;
    std::string m_transportProtocol; //!< the transport protocol to be used: TCP or UDP
};

/**
 * @ingroup applications-test
 * @ingroup tests
 *
 * Test that the packet size and packet arrival time is generated according to NGMN parameters
 */
class TrafficGeneratorNgmnVideoTestCase : public TestCase
{
  public:
    TrafficGeneratorNgmnVideoTestCase();
    ~TrafficGeneratorNgmnVideoTestCase() override;

  private:
    void DoRun() override;
};

/**
 * @ingroup applications-test
 * @ingroup tests
 *
 * Test that the packet size and packet arrival times that are generated correctly
 * according to the Fischer-Tippet distribution and are correctly configured
 * to reproduce the NGMN gaming models for the downlink and uplink
 */
class TrafficGeneratorNgmnGamingTestCase : public TestCase
{
  public:
    TrafficGeneratorNgmnGamingTestCase();
    ~TrafficGeneratorNgmnGamingTestCase() override;

  private:
    void DoRun() override;
};

/**
 * @ingroup applications-test
 * @ingroup tests
 *
 * Test that the average throughput is equal to the average VoIP throughput as
 * defined in NGMN white paper (RTP AMR 12.2)
 */
class TrafficGeneratorNgmnVoipTestCase : public TestCase
{
  public:
    TrafficGeneratorNgmnVoipTestCase(std::string transportProtocol);
    ~TrafficGeneratorNgmnVoipTestCase() override;

  private:
    void DoRun() override;
    std::string m_transportProtocol; //!< the transport protocol to be used: TCP or UDP
};

/**
 * @ingroup applications-test
 * @ingroup tests
 *
 * Test that the HTTP variables are generated according to NGMN/3GPP parameters
 */
class TrafficGeneratorThreeGppHttpTestCase : public TestCase
{
  public:
    TrafficGeneratorThreeGppHttpTestCase();
    ~TrafficGeneratorThreeGppHttpTestCase() override;

  private:
    void DoRun() override;
};

/**
 * @ingroup applications-test
 * @ingroup tests
 *
 * @brief Traffic generators test suite
 */
class TrafficGeneratorTestSuite : public TestSuite
{
  public:
    TrafficGeneratorTestSuite();
};

} // namespace ns3
#endif // TRAFFIC_GENERATOR_TEST_H
